/**
 * \file: grl_bitmap_decoder_util.h
 *
 * \version: $Id: grl_bitmap_decoder_util.h,v 1.11 2010/01/19 10:18:16 tkniep Exp $ 
 *
 * This header file declares the utility functions which implement common functions
 * that are used by the decoder modules.
 *
 * This file is for internal use in the Bitmap Decoder and should not be included
 * by any user application!
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/


#ifndef GRL_BITMAP_DECODER_UTIL_H
#define GRL_BITMAP_DECODER_UTIL_H

#ifdef __cplusplus
extern "C" {
#endif

#include <svg_typedef.h>
#include "svg_bitmap_decoder_err.h"

/*******************************************************************************************
 *   Macro definitions
 *******************************************************************************************/

/*
 * Names of DEVCONF keys
 */
#define GRL_BMPDEC_DEVKEY_RGB_TO_BGR            ((SVGChar*)(void*)"SVGBMPDEC_RGB_TO_BGR")
#define GRL_BMPDEC_DEVKEY_USE_JPEG              ((SVGChar*)(void*)"SVGBMPDEC_USE_JPEG")
#define GRL_BMPDEC_DEVKEY_USE_PNG               ((SVGChar*)(void*)"SVGBMPDEC_USE_PNG")
#define GRL_BMPDEC_DEVKEY_USE_FGS               ((SVGChar*)(void*)"SVGBMPDEC_USE_FGS")
#define GRL_BMPDEC_DEVKEY_PNG_DISP_GAMMA        ((SVGChar*)(void*)"SVGBMPDEC_PNG_DISP_GAMMA")
#define GRL_BMPDEC_DEVKEY_PNG_FILE_GAMMA        ((SVGChar*)(void*)"SVGBMPDEC_PNG_FILE_GAMMA")
#define GRL_BMPDEC_DEVKEY_QUEUE_SIZE            ((SVGChar*)(void*)"SVGBMPDEC_QUEUE_SIZE")
#define GRL_BMPDEC_DEVKEY_DEC_TSK_PRIO          ((SVGChar*)(void*)"SVGBMPDEC_DEC_TSK_PRIO")
#define GRL_BMPDEC_DEVKEY_DEC_TSK_STKSZ         ((SVGChar*)(void*)"SVGBMPDEC_DEC_TSK_STKSZ")
#define GRL_BMPDEC_DEVKEY_CTX_MAX               ((SVGChar*)(void*)"SVGBMPDEC_CTX_MAX")

/** Depth of ARGB8888 frame buffers in bytes */
#define GRL_BMPDEC_DEPTH_ARGB8888       (4)

/** color conversion from 8888 to 565 format*/
#define CHANGE_ARGB8888ToRGB565(x)                                  \
    (U16) (                                                         \
          (((U32)(x) & 0x00f80000) >> 8)                            \
        | (((U32)(x) & 0x0000fc00) >> 5)                            \
        | (((U32)(x) & 0x000000f8) >> 3)                            \
    )

/** color conversion from 565 to 8888 format*/
#define CHANGE_RGB565ToARGB8888(x)                                  \
    (U32) (                                                         \
          (((U32)(x) & 0x0000f800) << 8)                            \
        | (((U32)(x) & 0x000007e0) << 5)                            \
        | (((U32)(x) & 0x0000001f) << 3)                            \
    )

#define CHANGE_RGB565ToABGR8888(x)                                  \
    (U32) (                                                         \
          (((U32)(x) & 0x0000f800) >> 8)                            \
        | (((U32)(x) & 0x000007e0) << 5)                            \
        | (((U32)(x) & 0x0000001f) << 19)                            \
    )

/** Floats smaller than this are considered to be zero */
#define GRL_BMPDEC_NEAR_ZERO            (0.0001)

/* Dlt specific variables for ctx and debug level */
extern SVGUint8 grl_sbmp_debugLevel;
extern void *grl_sbmp_dlt_ctx;

/* Macros exported to svg font for dlt tracing */
#define SVG_BMP_F(args...) SVG_DLT_F(grl_sbmp_dlt_ctx, grl_sbmp_debugLevel, args)
#define SVG_BMP_E(args...) SVG_DLT_E(grl_sbmp_dlt_ctx, grl_sbmp_debugLevel, args)
#define SVG_BMP_W(args...) SVG_DLT_W(grl_sbmp_dlt_ctx, grl_sbmp_debugLevel, args)
#define SVG_BMP_U(args...) SVG_DLT_U(grl_sbmp_dlt_ctx, grl_sbmp_debugLevel, args)
#define SVG_BMP_D(args...) SVG_DLT_D(grl_sbmp_dlt_ctx, grl_sbmp_debugLevel, args)

/*******************************************************************************************
 *   Internal data type definitions
 *******************************************************************************************/

/**
 * Storage for configuration data read from the devconf file
 */
typedef struct GRL_BMPDEC_conf_
{
    SVGInt32    use_jpeg;       /**< Whether JPEG is supported                  */
    SVGInt32    use_png;        /**< Whether PNG is supported                   */
    SVGInt32	use_fgs;		/**< Whether CHN565 is supported				*/
    SVGInt32    png_disp_gamma; /**< PNG display gamma                          */
    SVGInt32    png_file_gamma; /**< PNG file gamme                             */
    SVGInt32    queue_size;     /**< Size of decoder queue                      */
    SVGInt32    dec_tsk_prio;   /**< Priority of decoder tasks                  */
    SVGInt32    dec_tsk_stksz;  /**< Stack size of decoder tasks                */
    SVGUint32   ctx_max;        /**< Maximum number of contexts                 */
    SVGInt32   rgb_to_bgr;		/**< internally a conversation from RGB to BGR should be done */
} GRL_BMPDEC_conf;

    
/*******************************************************************************************
 *   API function prototypes
 *******************************************************************************************/

/**
 * Set up the clipping area.
 * Depending on clip rect, position and src and dst sizes, the
 * clip rect and positions are updated such that everything fits
 * on the destination.
 * 
 * \param[in]       dst_area_width          Destination width
 * \param[in]       dst_area_height         Destination height
 * \param[in,out]   *p_dst_xpos             Pointer to destination position x
 * \param[in,out]   *p_dst_ypos             Pointer to destination position y
 * \param[in,out]   *source_clip            Pointer to source clip rect
 * \param[in]       src_width               Width of source image
 * \param[in]       src_height              Height of source image
 * \param[out]      *p_is_clipped           Flag to indicate that parts of image are clipped
 *
 */
SVGError GRL_BMPDEC_clipping_setup ( SVGInt32               dst_area_width,
                                     SVGInt32               dst_area_height,
                                     SVGInt32               *p_dst_xpos,
                                     SVGInt32               *p_dst_ypos,
                                     SVGBmpDecRect          *source_clip,
                                     SVGUint16              src_width,
                                     SVGUint16              src_height,
                                     SVGBoolean             *p_is_clipped );


/**
 * Read configuration from devconf file.
 * All devconf-configurable settings are read
 * and stored in a configuration structure for
 * later access.
 * 
 * \return      SVG_NO_ERROR            Configuration was read successfully
 * \return      SVG_NOT_EXISTENT        Some devconf entry was missing, default was used
 *
 */
SVGError GRL_BMPDEC_read_conf( void );


/**
 * Get configuration
 * Get the configuration which was read from devconf
 * at initialization time.
 *
 * \param[out]  config                  Pointer where configuration will be stored
 *
 * \return      SVG_NO_ERROR            Configuration returned successfully
 * \return      SVG_POINTER_NULL        config pointer was NULL
 */
SVGError GRL_BMPDEC_get_conf( GRL_BMPDEC_conf *config );


#ifdef __cplusplus
}
#endif

#endif /* GRL_BITMAP_DECODER_UTIL_H */
